<?php
// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: PUT, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if (!in_array($_SERVER['REQUEST_METHOD'], ['PUT', 'POST'])) {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

try {
    // Include database configuration
    require_once __DIR__ . '/../config.php';
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate input
    if (empty($input['id'])) {
        throw new Exception('ID is required');
    }
    
    // Sanitize input
    $id = (int)$input['id'];
    $updates = [];
    $params = [':id' => $id];
    
    // Build dynamic update query
    if (isset($input['url'])) {
        $url = filter_var($input['url'], FILTER_SANITIZE_URL);
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new Exception('Invalid URL format');
        }
        $updates[] = 'url = :url';
        $params[':url'] = $url;
    }
    
    if (isset($input['title'])) {
        $updates[] = 'title = :title';
        $params[':title'] = htmlspecialchars($input['title'], ENT_QUOTES, 'UTF-8') ?: 'New Website';
    }

    if (isset($input['description'])) {
        $updates[] = 'description = :description';
        $params[':description'] = htmlspecialchars($input['description'] ?? '', ENT_QUOTES, 'UTF-8');
    }
    
    // Add thumbnail_url handling
    if (array_key_exists('thumbnail_url', $input)) {
        if (!empty($input['thumbnail_url'])) {
            $thumbnail_url = filter_var($input['thumbnail_url'], FILTER_SANITIZE_URL);
            if (!filter_var($thumbnail_url, FILTER_VALIDATE_URL)) {
                throw new Exception('Invalid thumbnail URL format');
            }
            $updates[] = 'thumbnail_url = :thumbnail_url';
            $params[':thumbnail_url'] = $thumbnail_url;
        } else {
            // Allow setting thumbnail_url to null/empty
            $updates[] = 'thumbnail_url = NULL';
        }
    }
    
    if (isset($input['is_active'])) {
        $updates[] = 'is_active = :is_active';
        $params[':is_active'] = (bool)$input['is_active'] ? 1 : 0;
    }
    
    if (empty($updates)) {
        throw new Exception('No fields to update');
    }
    
    // Get database connection
    $pdo = getDBConnection();
    
    // Check if item exists
    $stmt = $pdo->prepare("SELECT id FROM webview_urls WHERE id = :id");
    $stmt->execute([':id' => $id]);
    
    if ($stmt->rowCount() === 0) {
        throw new Exception('WebView item not found');
    }
    
    // Build and execute update query
    $sql = "UPDATE webview_urls SET " . implode(', ', $updates) . ", updated_at = NOW() WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'WebView item updated successfully'
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}