<?php
require_once 'config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Load ad settings
$adConfigFile = 'ads_config.json';
$adSettings = [
    'showBannerAds' => true,
    'showInterstitialAds' => true,
    'bannerAdUnitId' => 'ca-app-pub-3940256099942544/6300978111',
    'interstitialAdUnitId' => 'ca-app-pub-3940256099942544/1033173712',
    'interstitialAdFrequency' => 3
];

// Load ad settings from file if it exists
if (file_exists($adConfigFile)) {
    $savedSettings = json_decode(file_get_contents($adConfigFile), true);
    if ($savedSettings) {
        $adSettings = array_merge($adSettings, $savedSettings);
    }
}

// Handle ad settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_ads') {
    $newSettings = [
        'showBannerAds' => isset($_POST['showBannerAds']),
        'showInterstitialAds' => isset($_POST['showInterstitialAds']),
        'bannerAdUnitId' => $_POST['bannerAdUnitId'] ?? $adSettings['bannerAdUnitId'],
        'interstitialAdUnitId' => $_POST['interstitialAdUnitId'] ?? $adSettings['interstitialAdUnitId'],
        'interstitialAdFrequency' => (int)($_POST['interstitialAdFrequency'] ?? $adSettings['interstitialAdFrequency'])
    ];
    
    file_put_contents($adConfigFile, json_encode($newSettings, JSON_PRETTY_PRINT));
    $adSettings = $newSettings;
    $message = "Ad settings updated successfully!";
    header("Location: " . $_SERVER['PHP_SELF'] . "?saved=1");
    exit;
}

$pdo = getDBConnection();
$message = isset($_GET['saved']) ? "Settings saved successfully!" : '';

// Handle file upload
function handleFileUpload($file, $existingFile = null) {
    $uploadDir = 'uploads/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }
    
    // If no new file is being uploaded, return the existing file path
    if (!isset($file['name']) || $file['error'] === UPLOAD_ERR_NO_FILE) {
        return $existingFile;
    }
    
    // Delete old file if it exists
    if ($existingFile && file_exists($existingFile)) {
        unlink($existingFile);
    }
    
    $fileName = uniqid() . '_' . basename($file['name']);
    $targetPath = $uploadDir . $fileName;
    
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        return $targetPath;
    }
    
    return $existingFile; // Return existing file if upload fails
}

// Handle Add/Edit/Delete actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'add':
                    $thumbnailPath = handleFileUpload($_FILES['thumbnail_file'] ?? null);
                    $isActive = isset($_POST['is_active']) ? 1 : 0;
                    $stmt = $pdo->prepare("INSERT INTO webview_urls (title, description, url, thumbnail_url, is_active) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([
                        $_POST['title'],
                        $_POST['description'] ?? '',
                        $_POST['url'],
                        $thumbnailPath,
                        $isActive
                    ]);
                    $message = "URL added successfully!";
                    break;
                
                case 'edit':
                    // Get existing URL data
                    $stmt = $pdo->prepare("SELECT thumbnail_url as thumbnail FROM webview_urls WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    $existingData = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    $thumbnailPath = handleFileUpload(
                        $_FILES['thumbnail_file'] ?? null,
                        $existingData['thumbnail'] ?? null
                    );
                    
                    // If no new file was uploaded and no existing thumbnail, use the URL if provided
                    if (empty($thumbnailPath) && !empty($_POST['thumbnail'])) {
                        $thumbnailPath = $_POST['thumbnail'];
                    }
                    
                    $isActive = isset($_POST['is_active']) ? 1 : 0;
                    $stmt = $pdo->prepare("UPDATE webview_urls SET title = ?, description = ?, url = ?, thumbnail_url = ?, is_active = ? WHERE id = ?");
                    $stmt->execute([
                        $_POST['title'],
                        $_POST['description'] ?? '',
                        $_POST['url'],
                        $thumbnailPath,
                        $isActive,
                        $_POST['id']
                    ]);
                    $message = "URL updated successfully!";
                    break;
                
                case 'delete':
                    $stmt = $pdo->prepare("DELETE FROM webview_urls WHERE id = ?");
                    $stmt->execute([$_POST['id']]);
                    $message = "URL deleted successfully!";
                    break;
            }
            header("Location: " . $_SERVER['PHP_SELF'] . "?saved=1");
            exit;
        } catch (PDOException $e) {
            $message = "Error: " . $e->getMessage();
        }
    }
}

// Fetch all URLs
$urls = $pdo->query("SELECT * FROM webview_urls ORDER BY id DESC")->fetchAll();

// Fetch analytics data if on analytics tab
$analyticsData = [];
$analyticsSummary = [
    'unique_users' => 0,
    'total_views' => 0,
    'total_clicks' => 0,
    'top_items' => [],
    'recent_activity' => []
];

if (file_exists('analytics_api.php')) {
    try {
        // Get summary data
        $analyticsSummary = json_decode(file_get_contents('analytics_api.php?action=summary'), true) ?? $analyticsSummary;
        
        // Get top items
        $topItems = json_decode(file_get_contents('analytics_api.php?action=top_items&limit=5'), true) ?? [];
        $analyticsSummary['top_items'] = $topItems;
        
        // Get recent activity
        $recentActivity = json_decode(file_get_contents('analytics_api.php?action=recent_activity&limit=5'), true) ?? [];
        $analyticsSummary['recent_activity'] = $recentActivity;
        
    } catch (Exception $e) {
        $message = "Error loading analytics: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.1.1/css/boxicons.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .navbar {
            box-shadow: 0 2px 4px rgba(0,0,0,.1);
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            margin-bottom: 1.5rem;
        }
        .card-header {
            border-radius: 10px 10px 0 0 !important;
            padding: 1rem 1.25rem;
        }
        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
        }
        .stat-card {
            background: #fff;
            border-radius: 8px;
            padding: 15px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            transition: transform 0.2s;
        }
        .stat-card:hover {
            transform: translateY(-3px);
        }
        .stat-number {
            display: block;
            font-size: 24px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        .stat-label {
            font-size: 13px;
            color: #7f8c8d;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        .nav-pills .nav-link.active {
            background-color: #0d6efd;
            font-weight: 500;
        }
        .nav-pills .nav-link {
            color: #495057;
        }
        .analytics-chart {
            background: #fff;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .recent-activity-item {
            padding: 10px 0;
            border-bottom: 1px solid #eee;
        }
        .recent-activity-item:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary mb-4">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php"><?php echo APP_NAME; ?></a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link <?php echo (!isset($_GET['tab']) || $_GET['tab'] !== 'analytics') ? 'active' : ''; ?>" 
                           href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-1"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'analytics') ? 'active' : ''; ?>" 
                           href="?tab=analytics">
                            <i class="fas fa-chart-line me-1"></i> Analytics
                        </a>
                    </li>
                </ul>
                <div class="d-flex">
                    <button class="btn btn-light me-2" data-bs-toggle="modal" data-bs-target="#adSettingsModal">
                        <i class="fas fa-ad me-1"></i> Ad Settings
                    </button>
                    <a href="logout.php" class="btn btn-outline-light">
                        <i class="fas fa-sign-out-alt me-1"></i> Logout
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <div class="container">
        <?php if ($message): ?>
            <div class="alert alert-info alert-dismissible fade show" role="alert">
                <?php echo htmlspecialchars($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <!-- Main Dashboard Content -->
        <?php if (!isset($_GET['tab']) || $_GET['tab'] !== 'analytics'): ?>
            <div class="row">
                <div class="col-md-4">
                    <!-- Add New URL Card -->
                    <div class="card mb-4">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">Add New URL</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="" enctype="multipart/form-data">
                                <input type="hidden" name="action" value="add">
                                <div class="mb-3">
                                    <label for="title" class="form-label">Title <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="title" name="title" required>
                                </div>
                                <div class="mb-3">
                                    <label for="description" class="form-label">Description</label>
                                    <textarea class="form-control" id="description" name="description" rows="2"></textarea>
                                    <div class="form-text">A brief description of the URL (optional)</div>
                                </div>
                                <div class="mb-3">
                                    <label for="url" class="form-label">URL <span class="text-danger">*</span></label>
                                    <input type="url" class="form-control" id="url" name="url" required>
                                </div>
                                <div class="mb-3">
                                    <label for="thumbnail_file" class="form-label">Thumbnail</label>
                                    <input type="file" class="form-control" id="thumbnail_file" name="thumbnail_file" accept="image/*">
                                    <div class="form-text">Or enter a URL below (leave empty to keep current)</div>
                                    <input type="url" class="form-control mt-2" id="thumbnail" name="thumbnail" placeholder="Or enter image URL">
                                </div>
                                <div class="mb-3 form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="is_active" name="is_active" checked>
                                    <label class="form-check-label" for="is_active">Active</label>
                                    <div class="form-text">Inactive URLs won't be shown in the app</div>
                                </div>
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-plus me-1"></i> Add URL
                                </button>
                            </form>
                        </div>
                    </div>

                    <!-- Quick Stats Card -->
                    <div class="card">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0">Quick Stats</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-12 mb-3">
                                    <div class="stat-card primary">
                                        <span class="stat-number"><?php echo $analyticsSummary['unique_users'] ?? 0; ?></span>
                                        <span class="stat-label">Total Users</span>
                                    </div>
                                </div>
                                <div class="col-6 mb-3">
                                    <div class="stat-card success">
                                        <span class="stat-number"><?php echo $analyticsSummary['total_views'] ?? 0; ?></span>
                                        <span class="stat-label">Total Views</span>
                                    </div>
                                </div>
                                <div class="col-6 mb-3">
                                    <div class="stat-card warning">
                                        <span class="stat-number"><?php echo $analyticsSummary['total_clicks'] ?? 0; ?></span>
                                        <span class="stat-label">Total Clicks</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-md-8">
                    <!-- URLs Table -->
                    <div class="card">
                        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">Manage URLs</h5>
                            <span class="badge bg-light text-dark"><?php echo count($urls); ?> URLs</span>
                        </div>
                        <div class="card-body p-0">
                            <?php if (empty($urls)): ?>
                                <div class="p-4 text-center text-muted">No URLs found. Add your first URL using the form.</div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-hover mb-0">
                                        <thead class="table-light">
                                            <tr>
                                                <th>ID</th>
                                                <th>Thumbnail</th>
                                                <th>Title</th>
                                                <th>Description</th>
                                                <th>URL</th>
                                                <th>Status</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($urls as $url): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($url['id']); ?></td>
                                                    <td>
                                                        <?php if (!empty($url['thumbnail'])): ?>
                                                            <img src="<?php echo htmlspecialchars($url['thumbnail']); ?>" alt="Thumbnail" style="width: 40px; height: 40px; object-fit: cover; border-radius: 4px;">
                                                        <?php else: ?>
                                                            <div class="bg-light d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; border-radius: 4px;">
                                                                <i class="fas fa-image text-muted"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td><?php echo htmlspecialchars($url['title']); ?></td>
                                                    <td>
                                                        <?php if (!empty($url['description'])): ?>
                                                            <span class="d-inline-block text-truncate" style="max-width: 200px;" title="<?php echo htmlspecialchars($url['description']); ?>">
                                                                <?php echo htmlspecialchars($url['description']); ?>
                                                            </span>
                                                        <?php else: ?>
                                                            <span class="text-muted">No description</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <a href="<?php echo htmlspecialchars($url['url']); ?>" target="_blank" class="text-truncate d-inline-block" style="max-width: 200px;" title="<?php echo htmlspecialchars($url['url']); ?>">
                                                            <?php echo htmlspecialchars($url['url']); ?>
                                                        </a>
                                                    </td>
                                                    <td>
                                                        <?php if ($url['is_active']): ?>
                                                            <span class="badge bg-success">Active</span>
                                                        <?php else: ?>
                                                            <span class="badge bg-secondary">Inactive</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <button class="btn btn-sm btn-outline-primary edit-url" 
                                                                data-id="<?php echo $url['id']; ?>"
                                                                data-title="<?php echo htmlspecialchars($url['title']); ?>"
                                                                data-description="<?php echo htmlspecialchars($url['description'] ?? ''); ?>"
                                                                data-url="<?php echo htmlspecialchars($url['url']); ?>"
                                                                data-thumbnail="<?php echo htmlspecialchars($url['thumbnail']); ?>"
                                                                data-is-active="<?php echo $url['is_active'] ? '1' : '0'; ?>">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <form method="POST" action="" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this URL?');">
                                                            <input type="hidden" name="action" value="delete">
                                                            <input type="hidden" name="id" value="<?php echo $url['id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-outline-danger">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Analytics Dashboard -->
        <?php if (isset($_GET['tab']) && $_GET['tab'] === 'analytics'): ?>
            <div class="row">
                <div class="col-12">
                    <div class="card mb-4">
                        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">Analytics Dashboard</h5>
                            <a href="dashboard.php" class="btn btn-light btn-sm">
                                <i class="fas fa-arrow-left me-1"></i> Back to Dashboard
                            </a>
                        </div>
                        <div class="card-body">
                            <!-- Summary Cards -->
                            <div class="row mb-4">
                                <div class="col-md-4 mb-3">
                                    <div class="card bg-primary text-white h-100">
                                        <div class="card-body text-center">
                                            <h2 class="display-4"><?php echo $analyticsSummary['unique_users'] ?? 0; ?></h2>
                                            <p class="mb-0">Total Users</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <div class="card bg-success text-white h-100">
                                        <div class="card-body text-center">
                                            <h2 class="display-4"><?php echo $analyticsSummary['total_views'] ?? 0; ?></h2>
                                            <p class="mb-0">Total Views</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <div class="card bg-info text-white h-100">
                                        <div class="card-body text-center">
                                            <h2 class="display-4"><?php echo $analyticsSummary['total_clicks'] ?? 0; ?></h2>
                                            <p class="mb-0">Total Clicks</p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Top Items -->
                            <div class="row">
                                <div class="col-md-6 mb-4">
                                    <div class="card h-100">
                                        <div class="card-header bg-light">
                                            <h6 class="mb-0">Top Viewed Items</h6>
                                        </div>
                                        <div class="card-body">
                                            <?php if (!empty($analyticsSummary['top_items'])): ?>
                                                <div class="list-group list-group-flush">
                                                    <?php foreach ($analyticsSummary['top_items'] as $item): ?>
                                                        <div class="list-group-item d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="mb-1"><?php echo htmlspecialchars($item['item_title'] ?? 'Untitled'); ?></h6>
                                                                <small class="text-muted"><?php echo htmlspecialchars($item['event_type']); ?></small>
                                                            </div>
                                                            <span class="badge bg-primary rounded-pill"><?php echo $item['count']; ?></span>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center text-muted py-4">No data available</div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <!-- Recent Activity -->
                                <div class="col-md-6 mb-4">
                                    <div class="card h-100">
                                        <div class="card-header bg-light">
                                            <h6 class="mb-0">Recent Activity</h6>
                                        </div>
                                        <div class="card-body">
                                            <?php if (!empty($analyticsSummary['recent_activity'])): ?>
                                                <div class="list-group list-group-flush">
                                                    <?php foreach ($analyticsSummary['recent_activity'] as $activity): ?>
                                                        <div class="list-group-item">
                                                            <div class="d-flex w-100 justify-content-between">
                                                                <h6 class="mb-1">
                                                                    <?php 
                                                                        $icon = 'fa-eye';
                                                                        if ($activity['event_type'] === 'webview_open') $icon = 'fa-external-link-alt';
                                                                        elseif ($activity['event_type'] === 'app_open') $icon = 'fa-mobile-alt';
                                                                    ?>
                                                                    <i class="fas <?php echo $icon; ?> me-2"></i>
                                                                    <?php echo ucfirst(str_replace('_', ' ', $activity['event_type'])); ?>
                                                                </h6>
                                                                <small class="text-muted">
                                                                    <?php 
                                                                        $date = new DateTime($activity['created_at']);
                                                                        echo $date->format('M j, Y g:i A');
                                                                    ?>
                                                                </small>
                                                            </div>
                                                            <p class="mb-1">
                                                                <?php echo !empty($activity['item_title']) ? 
                                                                    htmlspecialchars($activity['item_title']) : 
                                                                    'System Event'; 
                                                                ?>
                                                            </p>
                                                            <?php if (!empty($activity['device_info'])): ?>
                                                                <small class="text-muted">
                                                                    <i class="fas fa-info-circle me-1"></i>
                                                                    <?php echo htmlspecialchars($activity['device_info']); ?>
                                                                    <?php if (!empty($activity['app_version'])): ?>
                                                                        (v<?php echo htmlspecialchars($activity['app_version']); ?>)
                                                                    <?php endif; ?>
                                                                </small>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center text-muted py-4">No recent activity</div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Edit URL Modal -->
    <div class="modal fade" id="editUrlModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit URL</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" action="" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit-id">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="edit-title" class="form-label">Title <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="edit-title" name="title" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit-description" class="form-label">Description</label>
                            <textarea class="form-control" id="edit-description" name="description" rows="2"></textarea>
                            <div class="form-text">A brief description of the URL (optional)</div>
                        </div>
                        <div class="mb-3">
                            <label for="edit-url" class="form-label">URL <span class="text-danger">*</span></label>
                            <input type="url" class="form-control" id="edit-url" name="url" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit-thumbnail-file" class="form-label">Change Thumbnail</label>
                            <input type="file" class="form-control" id="edit-thumbnail-file" name="thumbnail_file" accept="image/*">
                            <div class="form-text">Or enter a URL below (leave empty to keep current)</div>
                            <input type="url" class="form-control mt-2" id="edit-thumbnail" name="thumbnail" placeholder="Enter new image URL">
                            <div class="mt-2" id="current-thumbnail-preview"></div>
                        </div>
                        <div class="mb-3 form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="edit-is_active" name="is_active" checked>
                            <label class="form-check-label" for="edit-is_active">Active</label>
                            <div class="form-text">Inactive URLs won't be shown in the app</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Ad Settings Modal -->
    <div class="modal fade" id="adSettingsModal" tabindex="-1" aria-labelledby="adSettingsModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="adSettingsModalLabel">AdMob Settings</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" action="" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="update_ads">
                    <div class="modal-body">
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="showBannerAds" name="showBannerAds" <?php echo $adSettings['showBannerAds'] ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="showBannerAds">Show Banner Ads</label>
                        </div>
                        
                        <div class="mb-3">
                            <label for="bannerAdUnitId" class="form-label">Banner Ad Unit ID</label>
                            <input type="text" class="form-control" id="bannerAdUnitId" name="bannerAdUnitId" 
                                   value="<?php echo htmlspecialchars($adSettings['bannerAdUnitId']); ?>">
                        </div>
                        
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="showInterstitialAds" name="showInterstitialAds" <?php echo $adSettings['showInterstitialAds'] ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="showInterstitialAds">Show Interstitial Ads</label>
                        </div>
                        
                        <div class="mb-3">
                            <label for="interstitialAdUnitId" class="form-label">Interstitial Ad Unit ID</label>
                            <input type="text" class="form-control" id="interstitialAdUnitId" name="interstitialAdUnitId" 
                                   value="<?php echo htmlspecialchars($adSettings['interstitialAdUnitId']); ?>">
                        </div>
                        
                        <div class="mb-3">
                            <label for="interstitialAdFrequency" class="form-label">Interstitial Frequency (number of clicks between ads)</label>
                            <input type="number" class="form-control" id="interstitialAdFrequency" name="interstitialAdFrequency" 
                                   min="1" value="<?php echo (int)$adSettings['interstitialAdFrequency']; ?>">
                        </div>
                        
                        <div class="alert alert-info">
                            <small>
                                <i class="fas fa-info-circle"></i> 
                                Note: These are test ad unit IDs. Replace them with your actual AdMob ad unit IDs before publishing your app.
                            </small>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary">Save changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        // Handle edit URL modal
        document.addEventListener('DOMContentLoaded', function() {
            // Edit URL modal
            const editUrlModal = new bootstrap.Modal(document.getElementById('editUrlModal'));
            const editButtons = document.querySelectorAll('.edit-url');
            
            editButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const id = this.getAttribute('data-id');
                    const title = this.getAttribute('data-title');
                    const description = this.getAttribute('data-description') || '';
                    const url = this.getAttribute('data-url');
                    const thumbnail = this.getAttribute('data-thumbnail') || '';
                    const isActive = this.getAttribute('data-is-active') === '1';
                    
                    document.getElementById('edit-id').value = id;
                    document.getElementById('edit-title').value = title;
                    document.getElementById('edit-description').value = description;
                    document.getElementById('edit-url').value = url;
                    document.getElementById('edit-thumbnail').value = ''; // Clear URL field
                    document.getElementById('edit-is_active').checked = isActive;
                    
                    // Show current thumbnail preview
                    const previewDiv = document.getElementById('current-thumbnail-preview');
                    if (thumbnail) {
                        previewDiv.innerHTML = `
                            <p class="mb-1"><small>Current Thumbnail:</small></p>
                            <img src="${thumbnail}" alt="Current Thumbnail" style="max-width: 100px; max-height: 60px; object-fit: contain; border: 1px solid #ddd; border-radius: 4px; padding: 2px;">
                        `;
                    } else {
                        previewDiv.innerHTML = '<p class="text-muted mb-0"><small>No thumbnail set</small></p>';
                    }
                    
                    editUrlModal.show();
                });
            });

            // Initialize tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });

        // Initialize activity chart if on analytics page
        <?php if (isset($_GET['tab']) && $_GET['tab'] === 'analytics' && !empty($analyticsData)): ?>
        document.addEventListener('DOMContentLoaded', function() {
            const ctx = document.getElementById('activityChart').getContext('2d');
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode(array_column($analyticsData, 'date')); ?>,
                    datasets: [{
                        label: 'Page Views',
                        data: <?php echo json_encode(array_column($analyticsData, 'views')); ?>,
                        borderColor: 'rgba(54, 162, 235, 1)',
                        backgroundColor: 'rgba(54, 162, 235, 0.1)',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'top',
                        },
                        title: {
                            display: true,
                            text: 'Page Views (Last 7 Days)'
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                stepSize: 1
                            }
                        }
                    }
                }
            });
        });
        <?php endif; ?>
    </script>
</body>
</html>
